/*==============================================================================
PART 3: Table 1 - Estimation Inputs
==============================================================================
Purpose:
Reproduce table 1 from the manuscript, which summarizes key estimation inputs:
- Panel A: Asset loss statistics for three key periods
- Panel B: Insured and uninsured deposit betas for three key periods
- Panel C: Insured and uninsured deposit costs for three key periods, by size

Input:
- $path_clean/analysis_ready_data.dta (Master dataset)

Output:
- $path_tables/table1_panel_a.tex
- $path_tables/table1_panel_b.tex
- $path_tables/table1_panel_c.tex

Last updated: July 03, 2025
==============================================================================*/
display "--- Creating Table 1 ---"

//===============================================================================
// Panel A: Asset Loss Statistics
//===============================================================================
// Purpose: Calculate and format summary statistics for asset losses for different
// periods and bank groups, outputting to a LaTeX table.

// Clear any previously stored estimation results
eststo clear

// Calculate summary statistics for all banks (Dec 2021)
eststo: estpost summarize loss if period == "dec2021" & !mi(DI_value) , d

// Calculate summary statistics for all banks (Feb 2023)
eststo: estpost summarize loss if period == "feb2023" & !mi(DI_value), d

// Calculate summary statistics for all banks (Feb 2024)
eststo: estpost summarize loss if period == "feb2024" & !mi(DI_value), d 

// Calculate summary statistics for large banks (assets >= $100 billion) (Dec 2021)
eststo: estpost summarize loss if period == "dec2021" & !mi(DI_value)  & assets_bill>=100, d

// Calculate summary statistics for large banks (assets >= $100 billion) (Feb 2023)
eststo: estpost summarize loss if period == "feb2023" & !mi(DI_value) & assets_bill>=100, d

// Calculate summary statistics for large banks (assets >= $100 billion) (Feb 2024)
eststo: estpost summarize loss if period == "feb2024" & !mi(DI_value) & assets_bill>=100, d

// Format and export the stored summary statistics to a LaTeX table
// The esttab command combines the results from the 6 eststo commands above
// into a single table with 6 columns (3 periods for all banks, 3 for large banks).
esttab using "$path_tables/table1_panel_a.tex",  main(mean) aux(sd) nonumber  ///
    prehead("\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "{\begin{tabular*}{0.9\textwidth}{@{\hskip\tabcolsep\extracolsep\fill}l*{6}{c}}" ///
    "\hline\hline" "&\multicolumn{3}{c}{All banks}" "&\multicolumn{3}{c}{Large banks}" "\\\cmidrule(lr){2-4}" "\cmidrule(lr){5-7}" "&\multicolumn{1}{c}{Dec 2021}&\multicolumn{1}{c}{Feb 2023}&\multicolumn{1}{c}{Feb 2024}&\multicolumn{1}{c}{Dec 2021}&\multicolumn{1}{c}{Feb 2023}&\multicolumn{1}{c}{Feb 2024}\\\cmidrule(lr){2-2}\cmidrule(lr){3-3}\cmidrule(lr){4-4}\cmidrule(lr){5-5}\cmidrule(lr){6-6}\cmidrule(lr){7-7}" "&\multicolumn{1}{c}{(1)}"   "&\multicolumn{1}{c}{(2)}"   "&\multicolumn{1}{c}{(3)}" "&\multicolumn{1}{c}{(4)}" "&\multicolumn{1}{c}{(5)}"  "&\multicolumn{1}{c}{(6)}"   "\\") ///
    width(0.8\textwidth) nogaps ///
    b(%8.2f) sfmt(%12.0fc %9.2fc) order(loss) /// /* Updated for long format */
    coeflabels(loss "Asset loss" _cons "Constant") ///
    noobs scalars("N Obs.") /// /* Include N (number of observations) scalar */
    msign($-$) replace nonotes nomtitles substitute(\_ _) starlevels(* 0.10 ** 0.05 *** 0.01) /// /* Standard formatting options */
    postfoot("\hline\hline \end{tabular*}}") // LaTeX postamble for the table

display "--- Table 1, panel a created: $path_tables/table1_panel_a.tex ---"

//===============================================================================
// Panel B: Insured & Uninsured Deposit Betas
//===============================================================================
//Purpose: Generate summary statistics for the decomposed insured and uninsured
// deposit betas for the three key periods.

eststo clear

// Summarize Dec 2021 betas
eststo: estpost summarize beta_ins beta_unins if period == "dec2021", d

// Summarize Feb 2023 betas
eststo: estpost summarize beta_ins beta_unins if period == "feb2023", d

// Summarize Feb 2024 betas
eststo: estpost summarize beta_ins beta_unins if period == "feb2024", d

esttab using "$path_tables/table1_panel_b.tex",  main(mean) aux(sd) replace ///
    prehead("{\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\begin{tabular*}{\textwidth}{@{\hskip\tabcolsep\extracolsep\fill}l*{3}{c}}" "\hline\hline"  " &\multicolumn{1}{c}{Dec 2021}" " &\multicolumn{1}{c}{Feb 2023}" "&\multicolumn{1}{c}{Feb 2024}" "\\\cmidrule(lr){2-2}" "\cmidrule(lr){3-3}" "\cmidrule(lr){4-4}") ///
    coeflabels(beta_ins "Insured deposit beta" beta_unins "Uninsured deposit beta") ///
    order(beta_ins beta_unins) /// /* Ensure correct row order */
    nonote label ///
    sfmt(%8.0g) nomtitles noobs scalars("N Obs.") /// /* Modified sfmt to format mean, sd, and N */
    msign($-$) alignment(c) ///
    postfoot("\hline\hline \end{tabular*}}")

display "--- Table 1, Panel B created: $path_tables/table1_panel_b.tex ---"

//===============================================================================
// Panel C: Insured & Uninsured deposit costs
//===============================================================================
// Purpose: Generate summary statistics for the insured and uninsured deposit costs
// for the three key periods, by size
eststo clear

texdoc init "$path_tables/table1_panel_c.tex", replace force

tex \begin{tabular*}{0.75\textwidth}{@{\hskip\tabcolsep\extracolsep\fill}l*{3}{c}}
tex \hline \hline

tex &\multicolumn{1}{c}{Insured} &\multicolumn{1}{c}{Uninsured} &\multicolumn{1}{c}{Overall} \\\cmidrule(lr){2-2} \cmidrule(lr){3-3} \cmidrule(lr){4-4} &\multicolumn{1}{c}{(1)} &\multicolumn{1}{c}{(2)} &\multicolumn{1}{c}{(3)} \\
tex \hline

foreach deptype in ins unins tot {
	summarize cost_`deptype' if period == "dec2021", d
	local mean_`deptype': di %6.3f r(mean)
	local sd_`deptype': di %6.3f r(sd)
	local sd_`deptype': subinstr local sd_`deptype' " " ""
}

tex All Banks & `mean_ins' & `mean_unins' & `mean_tot' \\
tex & (`sd_ins') & (`sd_unins') & (`sd_tot') \\

foreach sz in 1 2 3 4 {
	foreach deptype in ins unins tot {
	summarize cost_`deptype' if period == "dec2021" & sizeq==`sz', d
	local mean_`deptype'_size`sz': di %6.3f r(mean)
	local sd_`deptype'_size`sz': di %6.3f r(sd)
	local sd_`deptype'_size`sz': subinstr local sd_`deptype'_size`sz' " " ""
	}
}

tex Size & & & \\

foreach sz in 1 2 3 4 {
	tex \hspace{2em}`sz' & `mean_ins_size`sz'' & `mean_unins_size`sz'' & `mean_tot_size`sz'' \\
	tex & (`sd_ins_size`sz'') & (`sd_unins_size`sz'') & (`sd_tot_size`sz'') \\
}

tex \hline \hline
tex \end{tabular*}

texdoc close
display "--- Table 1, Panel C created: $path_tables/table1_panel_c.tex ---"

display "--- Table 1 finished ---"
